#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Fri Jan 21 22:27:14 2022

@author: max
"""



import sys
import numpy as np
import itertools





num_properties = 4
setsize = 3

def get_card_str(card : np.array) -> str:
    """
    

    Parameters
    ----------
    card : numpy.array
        A 1D numpy array of length num_properties.

    Returns
    -------
    s : str
        A string representation of the card.

    """
    s = ''
    for i in range(len(card)):
        s += chr(card[i])
    return s

def get_deck_str(cards) -> str:
    """
    

    Parameters
    ----------
    cards : numpy.array
        A 2D array describing the cards. Shape should be 
        (num_cards, num_properties)

    Returns
    -------
    s : str
        A string representation of the deck of cards

    """
    s = ''
    for card in cards:
        s += get_card_str(card)
        s += ' '
    return s


def solve_set(instr : str) -> list:
    """
    Solves the actual set problem on the numpy array cards. 

    Parameters
    ----------
    instr : str
        Input string to parse. Should be of length num_cards * num_properties.

    Returns
    -------
    solutions : list
        A list of tuples of length setsize each. Each tuple is a solution 
        of length setsize. The tuples contain the indicies of the used cards
        to create the solution.

    """
    assert(len(instr) % num_properties == 0)
    num_cards = len(instr) // num_properties
    cards = np.zeros( ( num_cards, num_properties), dtype=int)
    for i in range(0,len(instr),num_properties):
        for j in range(num_properties):
            cards[i//num_properties,j] = ord(instr[i+j])
    
    idxs_cards = list(range(len(cards)))
    solutions = []
    for perm in itertools.permutations(idxs_cards,setsize):
        for prop_id in range(num_properties):
            l = cards[perm,prop_id]
            unique = np.unique(l)
            if len(unique) != 1 and len(unique) < setsize:
                break
            else:
                if prop_id == num_properties-1:
                    solutions.append(perm)

    return solutions, cards

def argerror():
    """
    Prints an error message and quits the program.

    Returns
    -------
    None.

    """
    print(f'Please provide at least one card with {num_properties} properties!')
    exit(1)


"""
Run this if called as main file.
"""
if __name__ == "__main__":
    try:    
        instr = str(sys.argv[1])
    except:
        argerror()
        
    if len(instr) < num_properties:
        argerror()
    
    solutions, cards = solve_set(instr)
    
    if len(solutions) > 0:
        print(f'Found {len(solutions)} solutions:')
        for perm in solutions:
            print(get_deck_str(cards[list(perm)]))
        exit(0)
    else:
        print(f'Found no solution.')
        exit(1)
        

"""
Some unittest
"""
import unittest

class TestSetSolver(unittest.TestCase):
    
    def test_all_different(self):
        solutions, *_ = solve_set("1rno2gls3pfd")
        self.assertEqual(6,len(solutions))
        
        solutions, *_ = solve_set("qwerasdfzxcvghjk")
        self.assertEqual(24,len(solutions))
    
    def test_same(self):
        solutions, *_ = solve_set("1rno1gls1pfd")
        self.assertEqual(6,len(solutions))
        
        solutions, *_ = solve_set("1rno2rls3rfd")
        self.assertEqual(6,len(solutions))
        
        solutions, *_ = solve_set("1rfo2gfs3pfd")
        self.assertEqual(6,len(solutions))
        
        solutions, *_ = solve_set("1rns2gls3pfs")
        self.assertEqual(6,len(solutions))
    
    def test_solvable(self):
        solutions, *_ = solve_set("qwerasdrzxcr")
        self.assertTrue(len(solutions) > 0)
        
        solutions, *_ = solve_set("qsefzscfasdf")
        self.assertTrue(len(solutions) > 0)
        
        solutions, *_ = solve_set("qwerhjermnbvfber")
        self.assertTrue(len(solutions) > 0)
        

        solutions, *_ = solve_set("awerhwernwer")
        self.assertTrue(len(solutions) > 0)
        
    def test_no_solution(self):
        solutions, *_ = solve_set("qwerasdrzxcv")
        self.assertEqual(0,len(solutions))
        
        solutions, *_ = solve_set("qwertyurhjkfasdf")
        self.assertEqual(0,len(solutions))
        
        solutions, *_ = solve_set("qwerqwdfzxdf")
        self.assertEqual(0,len(solutions))

        solutions, *_ = solve_set("yuiohjklhjklyjilhuko")
        self.assertEqual(0,len(solutions))
        
    def test_only_one_card(self):
        solutions, *_ = solve_set("1rno")
        self.assertEqual(0,len(solutions))
        
    def test_only_two_card(self):
        solutions, *_ = solve_set("1rno1rno")
        self.assertEqual(0,len(solutions))
        
    
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        